import {
  createContext,
  useContext,
  useEffect,
  useState,
  ReactNode,
  useCallback,
  useMemo,
} from 'react';
import { catalogService, CatalogSummary } from '../services/catalogService';
import { SELECTED_CATALOG_STORAGE_KEY } from '../constants/catalog';

interface CatalogContextValue {
  catalogs: CatalogSummary[];
  selectedCatalogId: string | null;
  currentCatalog: CatalogSummary | undefined;
  setSelectedCatalogId: (catalogId: string) => void;
  refreshCatalogs: () => Promise<void>;
  isLoading: boolean;
  error: string | null;
}

const CatalogContext = createContext<CatalogContextValue | undefined>(undefined);

export const CatalogProvider = ({ children }: { children: ReactNode }) => {
  const [catalogs, setCatalogs] = useState<CatalogSummary[]>([]);
  const [selectedCatalogId, setSelectedCatalogIdState] = useState<string | null>(
    () => localStorage.getItem(SELECTED_CATALOG_STORAGE_KEY)
  );
  const [isLoading, setIsLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  const applyCatalogSelection = useCallback(
    (availableCatalogs: CatalogSummary[], existingSelection: string | null) => {
      if (!availableCatalogs.length) {
        setSelectedCatalogIdState(null);
        localStorage.removeItem(SELECTED_CATALOG_STORAGE_KEY);
        return null;
      }

      const selected =
        availableCatalogs.find(catalog => catalog.id === existingSelection) ??
        availableCatalogs.find(catalog => catalog.isDefault) ??
        availableCatalogs[0];

      if (selected) {
        setSelectedCatalogIdState(selected.id);
        localStorage.setItem(SELECTED_CATALOG_STORAGE_KEY, selected.id);
      }

      return selected?.id ?? null;
    },
    []
  );

  const loadCatalogs = useCallback(async () => {
    setIsLoading(true);
    setError(null);

    try {
      const response = await catalogService.getAll();
      setCatalogs(response);
    } catch (err) {
      console.error('Failed to load catalogs:', err);
      setError(err instanceof Error ? err.message : 'Failed to load catalogs');
    } finally {
      setIsLoading(false);
    }
  }, []);

  // Apply selection when catalogs are loaded
  useEffect(() => {
    if (catalogs.length > 0) {
      applyCatalogSelection(catalogs, selectedCatalogId);
    }
  }, [catalogs, applyCatalogSelection]); // selectedCatalogId omitted to prevent loop, as we only want to ensure selection on catalog load



  useEffect(() => {
    loadCatalogs();
  }, [loadCatalogs]);

  const setSelectedCatalogId = useCallback((catalogId: string) => {
    setSelectedCatalogIdState(catalogId);
    localStorage.setItem(SELECTED_CATALOG_STORAGE_KEY, catalogId);
  }, []);

  const currentCatalog = useMemo(
    () => catalogs.find(catalog => catalog.id === selectedCatalogId),
    [catalogs, selectedCatalogId]
  );

  const value = useMemo<CatalogContextValue>(
    () => ({
      catalogs,
      selectedCatalogId,
      currentCatalog,
      setSelectedCatalogId,
      refreshCatalogs: loadCatalogs,
      isLoading,
      error,
    }),
    [catalogs, currentCatalog, error, isLoading, loadCatalogs, selectedCatalogId, setSelectedCatalogId]
  );

  return <CatalogContext.Provider value={value}>{children}</CatalogContext.Provider>;
};

export const useCatalog = (): CatalogContextValue => {
  const context = useContext(CatalogContext);

  if (!context) {
    throw new Error('useCatalog must be used within a CatalogProvider');
  }

  return context;
};
