import bcrypt from 'bcryptjs';
import jwt from 'jsonwebtoken';
import prisma from '../config/database.js';
import { createError } from '../middleware/errorHandler.js';
// Generate JWT token
const generateToken = (userId, email, role) => {
    const secret = process.env.JWT_SECRET;
    const expiresIn = process.env.JWT_EXPIRES_IN || '7d';
    if (!secret) {
        throw createError('JWT secret not configured', 500);
    }
    return jwt.sign({ id: userId, email, role }, secret, { expiresIn });
};
// Generate refresh token
const generateRefreshToken = (userId) => {
    const secret = process.env.JWT_REFRESH_SECRET;
    const expiresIn = process.env.JWT_REFRESH_EXPIRES_IN || '30d';
    if (!secret) {
        throw createError('JWT refresh secret not configured', 500);
    }
    return jwt.sign({ id: userId }, secret, { expiresIn });
};
// Register new user
export const register = async (req, res, next) => {
    try {
        const { email, password, firstName, lastName, phone } = req.body;
        // Validation
        if (!email || !password) {
            throw createError('Email and password are required', 400);
        }
        if (password.length < 6) {
            throw createError('Password must be at least 6 characters', 400);
        }
        // Check if user exists
        const existingUser = await prisma.user.findUnique({
            where: { email },
        });
        if (existingUser) {
            throw createError('User already exists with this email', 409);
        }
        // Hash password
        const passwordHash = await bcrypt.hash(password, 10);
        // Create user
        const user = await prisma.user.create({
            data: {
                email,
                passwordHash,
                firstName,
                lastName,
                phone,
                role: 'CUSTOMER',
            },
            select: {
                id: true,
                email: true,
                firstName: true,
                lastName: true,
                phone: true,
                role: true,
                createdAt: true,
            },
        });
        // Generate tokens
        const token = generateToken(user.id, user.email, user.role);
        const refreshToken = generateRefreshToken(user.id);
        // Save refresh token
        await prisma.refreshToken.create({
            data: {
                token: refreshToken,
                userId: user.id,
                expiresAt: new Date(Date.now() + 30 * 24 * 60 * 60 * 1000), // 30 days
            },
        });
        res.status(201).json({
            success: true,
            data: {
                user,
                token,
                refreshToken,
            },
        });
    }
    catch (error) {
        next(error);
    }
};
// Login user
export const login = async (req, res, next) => {
    try {
        const { email, username, password } = req.body;
        // Validation
        if ((!email && !username) || !password) {
            throw createError('Email/Username and password are required', 400);
        }
        // Find user by email or username
        const user = await prisma.user.findFirst({
            where: {
                OR: [
                    email ? { email } : {},
                    username ? { username } : {},
                ],
            },
        });
        if (!user) {
            throw createError('Invalid credentials', 401);
        }
        // Check if user is active
        if (!user.isActive) {
            throw createError('Account is deactivated', 403);
        }
        // Verify password
        const isPasswordValid = await bcrypt.compare(password, user.passwordHash);
        if (!isPasswordValid) {
            throw createError('Invalid credentials', 401);
        }
        // Generate tokens
        const token = generateToken(user.id, user.email, user.role);
        const refreshToken = generateRefreshToken(user.id);
        // Save refresh token
        await prisma.refreshToken.create({
            data: {
                token: refreshToken,
                userId: user.id,
                expiresAt: new Date(Date.now() + 30 * 24 * 60 * 60 * 1000),
            },
        });
        res.json({
            success: true,
            data: {
                user: {
                    id: user.id,
                    email: user.email,
                    username: user.username,
                    firstName: user.firstName,
                    lastName: user.lastName,
                    phone: user.phone,
                    companyName: user.companyName,
                    customerType: user.customerType,
                    role: user.role,
                },
                token,
                refreshToken,
            },
        });
    }
    catch (error) {
        next(error);
    }
};
// Refresh token
export const refreshToken = async (req, res, next) => {
    try {
        const { refreshToken } = req.body;
        if (!refreshToken) {
            throw createError('Refresh token is required', 400);
        }
        // Verify refresh token
        const secret = process.env.JWT_REFRESH_SECRET;
        if (!secret) {
            throw createError('JWT refresh secret not configured', 500);
        }
        const decoded = jwt.verify(refreshToken, secret);
        // Check if refresh token exists in database
        const storedToken = await prisma.refreshToken.findUnique({
            where: { token: refreshToken },
            include: { user: true },
        });
        if (!storedToken || storedToken.expiresAt < new Date()) {
            throw createError('Invalid or expired refresh token', 401);
        }
        // Generate new tokens
        const newToken = generateToken(storedToken.user.id, storedToken.user.email, storedToken.user.role);
        const newRefreshToken = generateRefreshToken(storedToken.user.id);
        // Delete old refresh token and save new one
        await prisma.refreshToken.delete({
            where: { id: storedToken.id },
        });
        await prisma.refreshToken.create({
            data: {
                token: newRefreshToken,
                userId: storedToken.user.id,
                expiresAt: new Date(Date.now() + 30 * 24 * 60 * 60 * 1000),
            },
        });
        res.json({
            success: true,
            data: {
                token: newToken,
                refreshToken: newRefreshToken,
            },
        });
    }
    catch (error) {
        next(error);
    }
};
// Logout
export const logout = async (req, res, next) => {
    try {
        const { refreshToken } = req.body;
        if (refreshToken) {
            await prisma.refreshToken.deleteMany({
                where: { token: refreshToken },
            });
        }
        res.json({
            success: true,
            message: 'Logged out successfully',
        });
    }
    catch (error) {
        next(error);
    }
};
// Get current user
export const getCurrentUser = async (req, res, next) => {
    try {
        if (!req.user) {
            throw createError('Not authenticated', 401);
        }
        const user = await prisma.user.findUnique({
            where: { id: req.user.id },
            select: {
                id: true,
                email: true,
                username: true,
                firstName: true,
                lastName: true,
                phone: true,
                companyName: true,
                rneNumber: true,
                taxId: true,
                customerType: true,
                role: true,
                isActive: true,
                emailVerified: true,
                createdAt: true,
            },
        });
        if (!user) {
            throw createError('User not found', 404);
        }
        res.json({
            success: true,
            data: user,
        });
    }
    catch (error) {
        next(error);
    }
};
// Forgot password (placeholder - implement email sending)
export const forgotPassword = async (req, res, next) => {
    try {
        const { email } = req.body;
        if (!email) {
            throw createError('Email is required', 400);
        }
        const user = await prisma.user.findUnique({
            where: { email },
        });
        if (!user) {
            // Don't reveal if user exists
            res.json({
                success: true,
                message: 'If the email exists, a password reset link has been sent',
            });
            return;
        }
        // TODO: Generate reset token and send email
        // For now, just return success
        res.json({
            success: true,
            message: 'If the email exists, a password reset link has been sent',
        });
    }
    catch (error) {
        next(error);
    }
};
// Reset password (placeholder)
export const resetPassword = async (req, res, next) => {
    try {
        const { token, newPassword } = req.body;
        if (!token || !newPassword) {
            throw createError('Token and new password are required', 400);
        }
        if (newPassword.length < 6) {
            throw createError('Password must be at least 6 characters', 400);
        }
        // TODO: Verify reset token and update password
        res.json({
            success: true,
            message: 'Password reset successfully',
        });
    }
    catch (error) {
        next(error);
    }
};
//# sourceMappingURL=auth.controller.js.map