import { useState, useEffect } from 'react';
import { X, Download, Smartphone } from 'lucide-react';
import { useAuth } from '../context/AuthContext';
import {
    subscribeInstallAvailable,
    unsubscribeInstallAvailable,
    canPromptInstall,
    promptInstall,
    hasDismissedInstallPrompt
} from '../pwa/installPromptManager';

export default function PWAInstallPopup() {
    const { isAuthenticated } = useAuth();
    const [isVisible, setIsVisible] = useState(false);
    const [isInstalled, setIsInstalled] = useState(false);
    const [canInstall, setCanInstall] = useState(false);

    useEffect(() => {
        // Check if already installed
        if (window.matchMedia('(display-mode: standalone)').matches) {
            setIsInstalled(true);
        }

        // Check initial state
        setCanInstall(canPromptInstall());

        // Subscribe to updates
        const handleInstallAvailable = () => {
            setCanInstall(true);
        };

        subscribeInstallAvailable(handleInstallAvailable);

        return () => {
            unsubscribeInstallAvailable(handleInstallAvailable);
        };
    }, []);

    useEffect(() => {
        // Show popup if:
        // 1. User is logged in
        // 2. App is not already installed
        // 3. Browser allows installation (captured event)
        // 4. User hasn't dismissed it permanently (checked via manager)

        const dismissed = hasDismissedInstallPrompt();
        const localInstalled = localStorage.getItem('pwa_installed') === 'true';

        if (isAuthenticated && !isInstalled && !localInstalled && canInstall && !dismissed) {
            const timer = setTimeout(() => {
                setIsVisible(true);
            }, 1000);
            return () => clearTimeout(timer);
        }
    }, [isAuthenticated, isInstalled, canInstall]);

    const handleInstall = async () => {
        const outcome = await promptInstall();

        if (outcome === 'accepted') {
            console.log('User accepted the install prompt');
            localStorage.setItem('pwa_installed', 'true');
            setIsVisible(false);
        } else {
            console.log('User dismissed the install prompt');
        }
    };

    const handleClose = () => {
        setIsVisible(false);
    };

    if (!isVisible) return null;

    return (
        <div className="fixed inset-0 bg-black bg-opacity-50 z-[100] flex items-center justify-center p-4 animate-in fade-in duration-200">
            <div className="bg-white rounded-xl shadow-2xl max-w-sm w-full overflow-hidden relative animate-in zoom-in-95 duration-200">
                {/* Close Button */}
                <button
                    onClick={handleClose}
                    className="absolute top-3 right-3 p-1 text-gray-400 hover:text-gray-600 hover:bg-gray-100 rounded-full transition-colors"
                >
                    <X className="w-5 h-5" />
                </button>

                {/* Content */}
                <div className="p-6 text-center">
                    <div className="w-16 h-16 bg-blue-100 text-blue-600 rounded-full flex items-center justify-center mx-auto mb-4">
                        <Smartphone className="w-8 h-8" />
                    </div>

                    <h3 className="text-xl font-bold text-gray-900 mb-2">
                        Installer l'application
                    </h3>

                    <p className="text-gray-600 mb-6 leading-relaxed">
                        Installez notre application pour une expérience plus rapide et plus fluide.
                        Cliquez sur « Installer » pour ajouter l'application à votre appareil.
                    </p>

                    <button
                        onClick={handleInstall}
                        className="w-full bg-blue-600 text-white font-semibold py-3 px-4 rounded-lg hover:bg-blue-700 active:bg-blue-800 transition-colors flex items-center justify-center gap-2 shadow-lg shadow-blue-200"
                    >
                        <Download className="w-5 h-5" />
                        Installer
                    </button>
                </div>
            </div>
        </div>
    );
}
